#############################################################################
# Race Duration for LFS Lapper by Krayy
#############################################################################
# This mod allows a server admin to set a race duration for either a set number
# of minutes or an approximate number of Kms.
# This allows a greater level of granularity in race timings.
#############################################################################
#
# Syntax:
# !kms <kms>
#		This will set the race duration to the number of laps that is equal to
#		the specified distance divided by track length, rounded up.
#		
# !mins <mins> [<laps>]
#		This command will cause a race to set its last lap as the lap after
#		The specific <mins> has expired. The optional second parameter allows
#		you to set the number of extra laps at the end of the race
#		e.g. "!mins 10 3" will set a rcae to run for 10 mins then add 3 laps
#		to the lap counter
#
#############################################################################
# Ver 1.0.1 - 25 Mar 2011 Initial release
#############################################################################

const RD_LAPS		0;
const RD_KMS		1;
const RD_MINS		2;

CatchEvent OnLapperStart()
	GlobalVar $rdDuration; $rdDuration = 0;
	GlobalVar $rdDurationType; $rdDurationType = RD_LAPS;
	GlobalVar $raceTimerActive; $raceTimerActive = 0;
	GlobalVar $isTimedRace; $isTimedRace = 0;
	GlobalVar $rdLapCounter; $rdLapCounter = 0;
	GlobalVar $rdLapsToAdd; $rdLapsToAdd = 1;
EndCatchEvent

CatchEvent OnMSO( $userName, $text ) # Player event
	$idxOfFirstSpace = indexOf( $text, " ");
	IF( $idxOfFirstSpace == -1 ) THEN
	  $command = $text;
	  $argv = "";
	ELSE
	  $command = subStr( $text,0,$idxOfFirstSpace );
	  $argv = trim( subStr( $text,$idxOfFirstSpace ) );
	ENDIF

	SWITCH( $command )
		CASE "!mins":
			IF ( UserIsAdmin( $userName ) == 1 )
			THEN
				SetRaceMins($argv);
			ELSE
				privMsg( langEngine( "%{main_notadmin}%" ) );
			ENDIF
   		BREAK;
		CASE "!kms":
			IF ( UserIsAdmin( $userName ) == 1 )
			THEN
				SetRaceKms($argv);
			ELSE
				privMsg( langEngine( "%{main_notadmin}%" ) );
			ENDIF
   		BREAK;
	ENDSWITCH
EndCatchEvent

Sub SetRaceKms($argv)
	RemoveDelayedCommand( "raceTimer" );
	$myArgs = SplitTOArray( $argv , " " );

	# Reset the race duration and raceTimerActive flag
	$rdDuration = 0;
	$raceTimerActive = 0;
	$isTimedRace = 0;

	IF ( $myArgs[0] == "" )
	THEN
		$rdDurationType = RD_LAPS;
		globalMsg ( langEngine ( "%{raceDuration_Clear}%" ) );
		return();
	ENDIF

	IF ( IsNum($myArgs[0]) == TRUE )
	THEN
		$rdDurationType = RD_KMS;
		$rdDuration = (round( $myArgs[0] / GetLengthTrack(getLapperVar( "ShortTrackName" )) , 0 ) + 1);
		globalMsg ( langEngine ( "%{raceDuration_SetKms}%", $myArgs[0], $rdDuration  ) ) ;
		cmdLFS ("/laps " . $rdDuration);
	ELSE
		privMsg ( langEngine ( "%{raceDuration_NotNum}%", $myArgs[0] ) );
	ENDIF
EndSub

Sub SetRaceMins($argv)
	RemoveDelayedCommand( "raceTimer" );
	$myArgs = SplitTOArray( $argv , " " );

	$rdDuration = 0;
	$raceTimerActive = 0;
	$isTimedRace = 0;

	IF ( $myArgs[0] == "" )
	THEN
		$rdDurationType = RD_LAPS;
		globalMsg ( langEngine ( "%{raceDuration_Clear}%" ) );
		return();
	ENDIF
	
	IF ( $myArgs[1] == "" )
	THEN
		$rdLapsToAdd = 1;
	ELSE
		IF ( IsNum($myArgs[1]) == TRUE )
		THEN
			$rdLapsToAdd = $myArgs[1];
		ELSE
			privMsg ( langEngine ( "%{raceDuration_NotNum}%", $myArgs[1] ) );
			return();
		ENDIF
	ENDIF
	
	IF ( IsNum($myArgs[0]) == TRUE )
	THEN
		$rdDurationType = RD_MINS;
		$isTimedRace = 1;
		$rdDuration = ($myArgs[0] * 60) + 10;
		globalMsg ( langEngine ( "%{raceDuration_SetMins}%", $myArgs[0], $rdLapsToAdd  ) ) ;
		cmdLFS ("/hours 48");
	ELSE
		privMsg ( langEngine ( "%{raceDuration_NotNum}%", $myArgs[0] ) );
	ENDIF
EndSub

CatchEvent OnRaceStart ( $NumP ) # Lapper event
	RemoveDelayedCommand( "raceTimer" ); # Cancel the Race Timer if it's active
	IF ( $isTimedRace > 0 )
	THEN
		$raceTimerActive = 1;
		$rdLapCounter = 1;
		cmdLFS ("/hours 48");
		DelayedCommand ( 5, SendRaceTimerMessage );
		DelayedCommand ( "raceTimer", $rdDuration, RaceTimerComplete);
	ELSE
		RaceTimerTickerCloseAll();		# Close the event ticker
	ENDIF
EndCatchEvent

CatchEvent OnRaceEnd( ) # Lapper event
	RemoveDelayedCommand( "raceTimer" ); # Cancel the Race Timer if it's active
EndCatchEvent

CatchEvent OnLap( $userName ) # Player event
	IF ( $raceTimerActive > 0 )
	THEN
		IF ( ToNum(GetCurrentPlayerVar("Pos")) == 1 ) # Get the lead lap number if applicable
		THEN
			$rdLapCounter = ToNum(GetCurrentPlayerVar("LapsDone")) + 1;
#			privMsg ("rdLapCounter = " . $rdLapCounter);
		ENDIF
		RaceTimerTickerUpdate();	# Update the event ticker
	ENDIF
EndCatchEvent

Event OnSplit1( $userName ) # Player event
	RaceTimerTickerUpdate();	# Update the event ticker
EndEvent

Event OnSplit2( $userName ) # Player event 	
	RaceTimerTickerUpdate();	# Update the event ticker
EndEvent

Event OnSplit3( $userName ) # Player event
	RaceTimerTickerUpdate();	# Update the event ticker
EndEvent

CatchEvent OnPit( $userName )  # Player event
	RaceTimerTickerClose($userName);		# Close the event ticker
EndCatchEvent

CatchEvent OnLeaveRace( $userName )  # Player event
	RaceTimerTickerClose($userName);		# Close the event ticker
EndCatchEvent

Sub RaceTimerComplete ()
	$raceMsLeft = 0;
	$numLaps = $rdLapCounter + $rdLapsToAdd;
	$raceTimerActive = 0;
	globalRcm ( langEngine ( "%{raceDuration_EndMins}%", $numLaps  ));
	globalMsg ( langEngine ( "%{raceDuration_EndMins}%", $numLaps  ));
	cmdLFS ("/laps " . $numLaps);
	RemoveDelayedCommand( "raceTimer" );
EndSub

CatchEvent OnResult( $userName,$flagConfirm ) # Player event
	IF ( UserIsAdmin( $userName ) == 1 && $raceTimerActive > 0)
	THEN
		privMsg ( langEngine ( "%{raceDuration_End_Admin}%" ));
	ENDIF
EndCatchEvent

Sub getRacePercentComplete ()
	$RacePcLeft = 0;
	IF ( $rdDurationType == RD_MINS )
	THEN
		$RacePcLeft = round( ToNum( getLapperVar("ElapsedMS") / ($rdDuration * 1000) ) * 100, 0 );
	ELSE
		$RacePcLeft = Round(( ToNum(GetCurrentPlayerVar("LapsDone")) / getLapperVar("RaceLaps") ) * 100, 0);
	ENDIF
#	DEBUG ("Duration type: " . $rdDurationType . " and pct left is: " . $RacePcLeft );
	return ( $RacePcLeft );
EndSub

Sub RaceTimerTickerUpdate ()
	IF ( $raceTimerActive > 0 )
	THEN
		$raceTimeInMs = $rdDuration * 1000;
		$elapsedTime = round( getLapperVar("ElapsedMS"), 0 );
		
		IF ( $elapsedTime > $raceTimeInMs )
		THEN
			$raceMsLeft = 0;
		ELSE
			$raceMsLeft = $raceTimeInMs - $elapsedTime;
		ENDIF
		
		openPrivButton ( "raceDuration_Background",141,1,17,13,1,5,16,"");
		openPrivButton ( "raceDuration_Banner",142,2,15,4,1,5,35, langEngine ("%{raceDuration_Banner}%") );
		openPrivButton ( "raceDuration_Remaining",142,6,15,5,1,5,35, "^8" . NumToMS($raceMsLeft) );
	ENDIF
EndSub

Sub RaceTimerTickerCloseAll ()
	closeButtonRegexAll ("raceDuration_.*");
EndSub

Sub RaceTimerTickerClose ($userName)
	closeButtonRegex ($userName, "raceDuration_.*");
EndSub

Sub SendRaceTimerMessage()
	globalRcm(  langEngine ( "%{raceDuration_StartMins}%", round($rdDuration / 60,0 ) ) );
EndSub

#############################################################################

Lang "EN" # Race Events messages
	raceDuration_SetMins = "^2Race duration set to ^3{0}^2 minutes + ^3{1}^2 laps.";
	raceDuration_SetKms = "^2Race duration set to ^3{0}^2 kms, approximately ^3{1}^2 laps.";
	raceDuration_NotNum = "^2Race duration ^1not^2 set as ^3{0} ^2is not an integer.";
	raceDuration_Clear = "^3Race duration cleared and races are now in Laps.";
	raceDuration_Banner = "^3Time remaining:";
	raceDuration_StartMins = "^3This is a timed race and will run for {0} minutes.";
	raceDuration_EndMins = "^2Times up! Race will end on lap ^3{0}^2!";
	raceDuration_End_Admin = "^Race Timer active! Use !mins command to clear Race Timer for next race";
EndLang
